# ---------------------------------------------------------------------------------
# Program Name: getCTinfo.py
# Description :
# ClinicalTrial.gov APIサイトから試験番号、試験タイトル、試験要約を
# 取得するプログラム
# Author: JPMA TF5 2023
# ---------------------------------------------------------------------------------
# Libraries
import requests
import json
import openpyxl




#*********************************************
# Constant variables
#*********************************************
# ClinicalTrial.gobのAPIサイトURL
CTAPI_URL = 'https://clinicaltrials.gov/api/v2/studies'
# 取得情報数
PAGESIZE = 'pageSize'
# クエリ情報
QUERY = 'query.cond'
# 取得フィールド
FIELDS = 'fields'
# 次ページ存在するかどうかのkey名称
NEXTPAGE = 'nextPageToken'
# 次ページトークン設定用フィールド
NEXTPAGETOKEN = 'pageToken'



#*********************************************
# User setting variables
#*********************************************
# 関心のあるキーワード情報
# サンプルとして"osteogenesis imperfecta"を設定
KEYWORDS = 'osteogenesis imperfecta'
# 取得したいフィールド情報
# サンプルとして試験番号（NCTId）、試験タイトル（BriefTitle）、試験要約(BriefSummary)を設定
GETFIELDS = ['NCTId', 'BriefTitle', 'BriefSummary']
# 取得情報数（最大1000）
PSIZE = 10
# 出力ファイル名
RESOUT = r'Your path\result.xlsx'




#*********************************************
# Local function
#*********************************************
# Function Name: http_request
# Description:
#   APIサイトにリクエストを送信し、送信結果（json形式）を辞書型
#   に変換して戻す関数
# Parameter:
#   url : API request URL
#*********************************************
def http_request(url):
    
    # 引数で指定されたURLに接続する
    req = requests.get(url)
    
    # API送信結果（json形式）を辞書型に変換してメインに戻す
    return json.loads(req.text)





#*********************************************
# Main
#*********************************************
# API用URLの構成
param = '&'.join([QUERY + '=' + KEYWORDS,
                  PAGESIZE + '=' + str(PSIZE),
                  FIELDS + '=' + ','.join(GETFIELDS)
                  ])

API_URL = CTAPI_URL + '?' + param

print('---------------------------------------------')
print('ClinicalTrials.gov API URL:')
print(API_URL)
print('---------------------------------------------')



# API URLにHPPTリクエストを送信し、実行結果（json形式）を辞書型で取得する
dictCTInfo = http_request(API_URL)

# 検索結果の試験情報を配列に格納
studies = dictCTInfo['studies']


# 次ページ情報が存在する場合、情報を取得し続ける
while NEXTPAGE in dictCTInfo:
    
    # next page tokenの取得
    token = dictCTInfo['nextPageToken']
    
    # 次ページの情報を取得
    new_API_URL = API_URL + '&' + NEXTPAGETOKEN + '=' + token
    
    print('NOTE: Next page found ====>')
    print('ClinicalTrial.gov API URL:')
    print(new_API_URL)
    print('---------------------------------------------')

    # API URLにHPPTリクエストを送信し、実行結果（json形式）を辞書型で取得する
    dictCTInfo = http_request(new_API_URL)
    
    # 検索結果の試験情報を配列に追加する
    studies.extend(dictCTInfo['studies'])


# 各試験毎にNCTId, BriefTitle, BriefSummary情報を取得し、Excelファイルに出力する
wb = openpyxl.Workbook()
sheet = wb.worksheets[0]

# Column名の出力
sheet.append(['NCTId', 'BriefTitle', 'BriesSummary'])

for s in studies:
    
    # NCTId
    id = s['protocolSection']['identificationModule']['nctId']
    # BriefTitle
    btitle = s['protocolSection']['identificationModule']['briefTitle']
    # BriefSummary
    bsummary = s['protocolSection']['descriptionModule']['briefSummary']
    
    # 試験情報を配列に格納する
    sheet.append([id, btitle, bsummary])
    

wb.save(RESOUT)
wb.close()